/*
 * Copyright (c) 2005-2012, Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Module Name:    mfw_gst_v4lsrc.c
 *
 * Description:    Implementation of V4L Source Plugin for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */

/*
 * Changelog:
 *  1-Feb-2013:    Enhanced to support different RGB and YUV color formats,
 *                 configurable through plugin properties.
 *
 *  19-Dec-2013:   Added check for V4L2_BUF_FLAG_ERROR buffer flag and signal
 *                 will be emitted if flag is set to notify Gstreamer framework.
 *
 *  9-Jan-2014:    Enhanced to support different picture enhancement parameters,
 *                 configurable through plugin properties.
 *
 *  20-Jan-2014:   Added SIGNAL_V4L2_BUF_FLAG_OK signal to recover from error
 *                 state and to notify Gstreamer framework.
 *
 *  13-May-2014:   Added cropping feature to configure left, top, width and
 *                 height values through VIDIOC_S_CROP ioctl call
 *
 *  02-Feb-2016:   Update picture enhancement parameters min and max value.
 *                 Enhanced to support Cb and Cr (Chroma components),
 *                 configurable through plugin properties.
 *
 *  05-Apr-2016:   Pass appropriate GstBuffer flag to inform application
 *                 whether the captured buffer data is correct or corrupted.
 */


/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <gst/gst.h>
#include <gst/base/gstpushsrc.h>
#include <gst/interfaces/propertyprobe.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <stdbool.h>
#include <linux/videodev2.h>
#include <linux/v4l2-subdev.h>
#include <sys/mman.h>
#include <string.h>
#include <gst/video/video.h>
#include <media/imx6.h>
#include <media/adv718x.h>
#include "mfw_gst_v4lsrc.h"

#include "mfw_gst_utils.h"
#include "gstbufmeta.h"

//#define GST_DEBUG g_print

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
/* None */
#define DEFAULT_QUEUE_SIZE 6
#define NO_BUFFER_WAIT_COUNT   200
#define NO_BUFFER_WAIT_TIME_MS 5

#define MAX_ERRMEM_MESSAGE_LENGTH 256

#define SYSPATH_TEMPLATE "/sys/class/video4linux%s/name"

#define MAX_DEVICE_NAME_LENGTH          256
#define MAX_PATH_LENGTH                 256
#define DEINTERLACE_DEVICE_NAME         "vdic"
#define NULL_DEVICE_NAME                "NULL"

static void log_errmem(const char *devicename, const char *message, ...)
{
  va_list args;
  int ret = 0;
  int fd = -1;
  char *msg = NULL;

  if (devicename == NULL)
  {
    devicename = NULL_DEVICE_NAME;
  }

  fd = open("/dev/errmem", O_WRONLY);
  if (fd >= 0)
  {
    msg = (char *)calloc(MAX_ERRMEM_MESSAGE_LENGTH, sizeof(char));
    if (msg)
    {        
        {
          char *name_path = NULL;
          char *name = NULL;
          gsize length = 0;
          GError *error = NULL;

          ret = snprintf (msg, MAX_ERRMEM_MESSAGE_LENGTH,
                    "MFW_V4LSRC: Using capture device %s\n", devicename);

          if (ret > 0 && ret < MAX_ERRMEM_MESSAGE_LENGTH)
          {
            write(fd, msg, sizeof(char) * (ret + 1));
          }
          else if (ret > -1)
          {
            write(fd, msg, MAX_ERRMEM_MESSAGE_LENGTH);
          }
          else
          {
            const char errmsg[] = "MFW_V4LSRC: Failed to vsnprintf log message\n";
            write(fd, errmsg, sizeof(errmsg));
          }

          name_path = g_strdup_printf(SYSPATH_TEMPLATE, strrchr(devicename, '/'));
          g_file_get_contents(name_path, &name, &length, &error);
          g_free(name_path);
          if (error != NULL)
          {
              ret = snprintf(msg, MAX_ERRMEM_MESSAGE_LENGTH,
                             "MFW_V4LSRC: Failed to get name for device: %s\n",
                             error->message);
            g_error_free(error);
          }
          else
          {
              ret = snprintf(msg, MAX_ERRMEM_MESSAGE_LENGTH,
                             "MFW_V4LSRC: Capture device has name %s\n", name);
              g_free(name);
          }
        }

        if (ret > 0 && ret < MAX_ERRMEM_MESSAGE_LENGTH)
        {
          write(fd, msg, sizeof(char) * (ret + 1));
        }
        else if (ret > -1)
        {
          write(fd, msg, MAX_ERRMEM_MESSAGE_LENGTH);
        }
        else
        {
          const char errmsg[] = "MFW_V4LSRC: Failed to vsnprintf log message\n";
          write(fd, errmsg, sizeof(errmsg));
        }

      va_start(args, message);
      ret = vsnprintf(msg, MAX_ERRMEM_MESSAGE_LENGTH, message, args);
      va_end(args);

      if (ret > 0 && ret < MAX_ERRMEM_MESSAGE_LENGTH)
      {
        write(fd, msg, sizeof(char) * (ret + 1));
      }
      else if (ret > -1)
      {
        write(fd, msg, MAX_ERRMEM_MESSAGE_LENGTH);
      }
      else
      {
        const char errmsg[] = "MFW_V4LSRC: Failed to vsnprintf log message\n";
        write(fd, errmsg, sizeof(errmsg));
      }
      free(msg);
    }
    else
    {
      const char errmsg[] = "MFW_V4LSRC: Failed to allocate memory for message\n";
      write(fd, errmsg, sizeof(errmsg));
    }
    close(fd);
  }
}


#define timed_ioctl(a,b,c) __timed_ioctl(v4l_src->devicename, __FUNCTION__, "" # b, (a), (b), (c))

static inline __timed_ioctl(const char *devicename,
                            const char *fn,
                            const char *ioctl_name,
                            int fd,
                            unsigned long request,
                            void* data)
{
    int res = 0;
    res = ioctl(fd, request, data);

    if (res < 0)
    {
        log_errmem(devicename, "MFW_V4lSRC ioctl %s in %s failed\n", ioctl_name, fn);
    }

    return res;
}


/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/

enum
{
  MFW_V4L_SRC_0,
  MFW_V4L_SRC_WIDTH,
  MFW_V4L_SRC_HEIGHT,
  MFW_V4L_SRC_INPUT,
  MFW_V4L_SRC_ROTATE,
  MFW_V4L_SRC_HFLIP,
  MFW_V4L_SRC_VFLIP,
  MFW_V4L_SRC_PREVIEW,
  MFW_V4L_SRC_PREVIEW_WIDTH,
  MFW_V4L_SRC_PREVIEW_TOP,
  MFW_V4L_SRC_PREVIEW_LEFT,
  MFW_V4L_SRC_PREVIEW_HEIGHT,
  MFW_V4L_SRC_PREVIEW_COLOR_KEY,
  MFW_V4L_SRC_PREVIEW_GALPHA,
  MFW_V4L_SRC_CROP_LEFT,   /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_TOP,    /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_WIDTH,  /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_HEIGHT, /* Enhanced to support cropping feature */
  MFW_V4L_SRC_FRAMERATE_NUM,
  MFW_V4L_SRC_FRAMERATE_DEN,
  MFW_V4L_SRC_OUTPUT_FRAMERATE_NUM,
  MFW_V4L_SRC_OUTPUT_FRAMERATE_DEN,
  MFW_V4L_SRC_DEVICE,
  MFW_V4L_SRC_QUEUE_SIZE,
  MFW_V4L_SRC_PIXEL_FMT, /* Enhanced to support diff RGB and YUV color-formats */
  MFW_V4L_SRC_BRIGHTNESS, /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_CONTRAST,   /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_SATURATION, /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_HUE,        /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_MOTION,     /* Enhanced to set deinterlace motion compensation parameters */
  MFW_V4L_SRC_CHROMA_BLUE,	/* Enhanced to support chroma components */
  MFW_V4L_SRC_CHROMA_RED,	/* Enhanced to support chroma components */
  MFW_V4L_STREAM_GOOD_BUFFER
};

enum
{
  SIGNAL_V4L2_BUF_FLAG_ERROR,
  SIGNAL_V4L2_BUF_FLAG_OK,
  SIGNAL_V4L2_BUF_FLAG_NONE
};

static guint gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_NONE] = { 0 };

static GstStaticPadTemplate src_factory =
  GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS(
        GST_VIDEO_CAPS_RGB_16 ";"
        GST_VIDEO_CAPS_BGR ";"
        GST_VIDEO_CAPS_RGB ";"
        GST_VIDEO_CAPS_BGRx ";"
        GST_VIDEO_CAPS_RGBx ";"
        GST_VIDEO_CAPS_YUV ("422P") ";"
        GST_VIDEO_CAPS_YUV ("UYVY") ";"
        GST_VIDEO_CAPS_YUV ("YUY2") ";"
        GST_VIDEO_CAPS_YUV ("I420") ";"
        GST_VIDEO_CAPS_YUV ("NV12") ";"
        )
  );


/*=============================================================================
                              LOCAL MACROS
=============================================================================*/
/*
 * The new i.mx6 capture driver implements a private ioctl to retrieve a
 * v4l2 buffer's physaddr. The old capture driver hacks VIDIOC_QUERYBUF
 * to do this.
 */
#define VIDIOC_IMX6_GET_PHYS \
        _IOWR('i', BASE_VIDIOC_PRIVATE+0, struct v4l2_buffer)

/* used for debugging */
#define GST_CAT_DEFAULT mfw_gst_v4lsrc_debug

/* Enhanced to support diff RGB and YUV color-formats */  
#define G_MAX_PIX_FMT	9

/* Enhanced to support picture enhancement parameters */
#define DEFAULT_BRIGHTNESS_VAL	0
#define DEFAULT_CONTRAST_VAL	128
#define DEFAULT_SATURATION_VAL	128
#define DEFAULT_HUE_VAL			0

#define MIN_BRIGHTNESS_VAL		0
#define MIN_CONTRAST_VAL		0
#define MIN_SATURATION_VAL		0
#define MIN_HUE_VAL				-127

#define MAX_BRIGHTNESS_VAL		255
#define MAX_CONTRAST_VAL		255
#define MAX_SATURATION_VAL		255
#define MAX_HUE_VAL				128

/* Enhanced to set deinterlace motion compensation parameters */
#define DEFAULT_MOTION_VAL 		1
#define G_MAX_MOTION_VAL		3

/* Enhanced to support chroma components */
#define DEFAULT_CHROMA_BLUE_VAL	0
#define DEFAULT_CHROMA_RED_VAL	0

#define MIN_CHROMA_BLUE_VAL		-128
#define MIN_CHROMA_RED_VAL		-128

#define MAX_CHROMA_BLUE_VAL		127
#define MAX_CHROMA_RED_VAL		127

/*=============================================================================
                             STATIC VARIABLES
=============================================================================*/

/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/
/* None */

/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC (mfw_gst_v4lsrc_debug);
static void mfw_gst_v4lsrc_buffer_class_init (gpointer g_class,
    gpointer class_data);
static void mfw_gst_v4lsrc_buffer_init (GTypeInstance * instance,
    gpointer g_class);
static void mfw_gst_v4lsrc_buffer_finalize (MFWGstV4LSrcBuffer * v4lsrc_buffer);
static void mfw_gst_v4lsrc_fixate (GstPad * pad, GstCaps * caps);
static GstCaps *mfw_gst_v4lsrc_get_caps (GstBaseSrc * src);
static GstFlowReturn mfw_gst_v4lsrc_create (GstPushSrc * src, GstBuffer ** buf);
static GstBuffer *mfw_gst_v4lsrc_buffer_new (MFWGstV4LSrc * v4l_src);
static GstStateChangeReturn mfw_gst_v4lsrc_change_state (GstElement * element,
    GstStateChange transition);
static gboolean unlock (GstBaseSrc *src);
static gboolean mfw_gst_v4lsrc_stop (GstBaseSrc * src);
static gboolean mfw_gst_v4lsrc_start (GstBaseSrc * src);
static gboolean mfw_gst_v4lsrc_set_caps (GstBaseSrc * src, GstCaps * caps);
static void mfw_gst_v4lsrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void mfw_gst_v4lsrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static gint mfw_gst_v4lsrc_capture_setup (MFWGstV4LSrc * v4l_src);
static gint mfw_gst_v4lsrc_stop_capturing (MFWGstV4LSrc * v4l_src);
static gint mfw_gst_v4lsrc_start_capturing (MFWGstV4LSrc * v4l_src);


GST_BOILERPLATE (MFWGstV4LSrc, mfw_gst_v4lsrc, GstPushSrc, GST_TYPE_PUSH_SRC);

static gboolean is_video_std(MFWGstV4LSrc * v4l_src)
{
	return v4l_src->std_id != V4L2_STD_ALL;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_get_type    

DESCRIPTION:        This funtion registers the  buffer type on to the V4L Source plugin
             
ARGUMENTS PASSED:   void   

RETURN VALUE:       Return the registered buffer type
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
GType
mfw_gst_v4lsrc_buffer_get_type (void)
{
  static GType v4lsrc_buffer_type;

  if (G_UNLIKELY (v4lsrc_buffer_type == 0)) {
    static const GTypeInfo v4lsrc_buffer_info = {
      sizeof (GstBufferClass),
      NULL,
      NULL,
      mfw_gst_v4lsrc_buffer_class_init,
      NULL,
      NULL,
      sizeof (MFWGstV4LSrcBuffer),
      0,
      mfw_gst_v4lsrc_buffer_init,
      NULL
    };
    v4lsrc_buffer_type = g_type_register_static (GST_TYPE_BUFFER,
        "MFWGstV4LSrcBuffer", &v4lsrc_buffer_info, 0);
  }
  return v4lsrc_buffer_type;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_class_init    

DESCRIPTION:   This funtion registers the  funtions used by the 
                buffer class of the V4l source plug-in
             
ARGUMENTS PASSED:
        g_class        -   class from which the mini objext is derived
        class_data     -   global class data

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_class_init (gpointer g_class, gpointer class_data)
{
  GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS (g_class);

  mini_object_class->finalize = (GstMiniObjectFinalizeFunction)
      mfw_gst_v4lsrc_buffer_finalize;
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_buffer_init    

DESCRIPTION:   This funtion initialises the buffer class of the V4l source plug-in
             
ARGUMENTS PASSED:
        instance       -   pointer to buffer instance
        g_class        -   global pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_init (GTypeInstance * instance, gpointer g_class)
{

}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_buffer_finalize    

DESCRIPTION:   This function is invoked whenever the buffer object belonging 
               to the V4L Source buffer glass is tried to un-refrenced. Here 
               only the refernce count of the buffer object is increased without 
               freeing the memory allocated.

ARGUMENTS PASSED:
        v4lsrc_buffer -   pointer to V4L sou4rce buffer class

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_finalize (MFWGstV4LSrcBuffer * v4lsrc_buffer)
{
  MFWGstV4LSrc *v4l_src;
  gint num;
  GstBuffer *buf;
  struct v4l2_buffer v4lbuf;

  v4l_src = v4lsrc_buffer->v4lsrccontext;
  if (v4l_src->start) {
    num = v4lsrc_buffer->num;


    buf = (GstBuffer *) (v4l_src->buffers[num]);
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_LAST);

    g_mutex_lock (v4l_src->pool_lock);
    if (g_list_find (v4l_src->free_pool, (gpointer) (num)))
        GST_WARNING ("something wrong here, v4l buffer index:%d already in queue",
                num);
    else
        GST_LOG ("v4l buffer index:%d will be push in pool", num);
    g_mutex_unlock (v4l_src->pool_lock);

    memset (&v4lbuf, 0, sizeof (v4lbuf));
    v4lbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    v4lbuf.memory = V4L2_MEMORY_MMAP;
    v4lbuf.index = num;

    if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_QBUF, &v4lbuf) < 0) {
      GST_ERROR (">>V4L_SRC: VIDIOC_QBUF failed");
      return;
    }

    g_mutex_lock (v4l_src->pool_lock);
    v4l_src->free_pool = g_list_append (v4l_src->free_pool, (gpointer) num);
    g_mutex_unlock (v4l_src->pool_lock);
    GST_LOG_OBJECT (v4l_src, "freeing buffer %p for frame %d", v4lsrc_buffer,
        num);
    gst_buffer_ref (GST_BUFFER_CAST (v4lsrc_buffer));
  } else {
    gint index;

    GST_LOG ("free buffer %d\n", v4lsrc_buffer->num);

    buf = (GstBuffer *)v4lsrc_buffer;

    /*
     * gstbuffer and the meta-data was not freed in
     * mfw_gst_v4lsrc_stop_capturing API if the buffer or metadata
     * was being used by other sink. In this case, the gstbuffer will
     * be unrefered by the other sink and the meta data has to be freed
     * here
     */
    index = G_N_ELEMENTS (buf->_gst_reserved) - 1;
    if (buf->_gst_reserved[index]) {
        GST_DEBUG ("Meta data for buffer:0x%x is freed", buf);
        gst_buffer_meta_free (buf->_gst_reserved[index]);
    }
  }
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_start_capturing    
        
DESCRIPTION:   This function triggers the V4L Driver to start Capturing

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_start_capturing (MFWGstV4LSrc * v4l_src)
{
	guint i;
	struct v4l2_buffer physbuf;
	struct v4l2_buffer *buf;
	MFWGstV4LSrcBuffer *v4lsrc_buf = NULL;
	enum v4l2_buf_type type;
	int loop_num =  BUFF_ALLOC_OR_FREE;
	v4l_src->loop_success = -1;
	v4l_src->num_buf_created = 0;

	while (loop_num < (STREAM_ON_OR_OFF + 1))
	{
		switch (loop_num)
		{
			case BUFF_ALLOC_OR_FREE:
				GST_DEBUG ("BUFF_ALLOC \n");
				v4l_src->buffers = g_malloc (v4l_src->queue_size * sizeof (GstBuffer *));
				if (v4l_src->buffers == NULL)
				{
					GST_ERROR (">>V4L_SRC: malloc failed");
					return -1;
				}
				break;
			case MAP_BUF_OR_UNMAP:
				GST_DEBUG ("MAP_BUF \n");
				for (i = 0; i < v4l_src->queue_size; i++)
				{
					v4lsrc_buf =
							(MFWGstV4LSrcBuffer *) gst_mini_object_new (MFW_GST_TYPE_V4LSRC_BUFFER);
					GST_DEBUG ("BUFFER LOOP %d \n",i);
					v4lsrc_buf->num = i;
					v4lsrc_buf->v4lsrccontext = v4l_src;
					/* v4l2_buffer initialization */
					buf = &v4lsrc_buf->v4l2_buf;
					buf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
					buf->memory = V4L2_MEMORY_MMAP;
					buf->index = i;
					if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_QUERYBUF, buf) < 0)
					{
						GST_ERROR (">>V4L_SRC: VIDIOC_QUERYBUF error");
						return -1;
					}

					physbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
					physbuf.index = i;
					if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_IMX6_GET_PHYS, &physbuf) < 0)
					{
						GST_ERROR ("VIDIOC_IMX6_GET_PHYS failed, %s", strerror (errno));
						return -1;
					}
					/* GstBuffer initialization */
					v4l_src->buffers[i] = (GstBuffer *) v4lsrc_buf;
					GST_BUFFER_SIZE (v4l_src->buffers[i]) = buf->length;
					GST_BUFFER_OFFSET (v4l_src->buffers[i]) = (size_t) buf->m.offset;
					GST_BUFFER_DATA (v4l_src->buffers[i]) = mmap (NULL,
							GST_BUFFER_SIZE (v4l_src->buffers[i]),
							PROT_READ | PROT_WRITE, MAP_SHARED,
							v4l_src->fd_v4l, GST_BUFFER_OFFSET (v4l_src->buffers[i]));
					memset (GST_BUFFER_DATA (v4l_src->buffers[i]), 0xFF,
							GST_BUFFER_SIZE (v4l_src->buffers[i]));
					{
						gint index;
						GstBufferMeta *meta;
						index = G_N_ELEMENTS (v4l_src->buffers[i]->_gst_reserved) - 1;
						meta = gst_buffer_meta_new ();
						meta->physical_data = (gpointer) (physbuf.m.offset);
						v4l_src->buffers[i]->_gst_reserved[index] = meta;
					}

					v4l_src->num_buf_created = i + 1;
					if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_QBUF, buf) < 0)
					{
						GST_ERROR (">>V4L_SRC: VIDIOC_QBUF error");
						return -1;
					}
					v4l_src->free_pool =
							g_list_append (v4l_src->free_pool, (gpointer) buf->index);
				}
				break;
			case CREATE_MUTEX_OR_DESTROY:
				GST_DEBUG ("CREATE_MUTEX \n");
				v4l_src->pool_lock = g_mutex_new ();
				break;
			case STREAM_ON_OR_OFF:
				GST_DEBUG ("STREAM_ON \n");
				/* Switch ON the capture device */
				type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
				if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_STREAMON, &type) < 0)
				{
					GST_ERROR (">>V4L_SRC: VIDIOC_STREAMON error");
					return -1;
				}
				v4l_src->start = TRUE;
				break;
			default:
				break;
		}
		v4l_src->loop_success = loop_num;
		loop_num = loop_num + 1;
	}

	v4l_src->time_per_frame =
			gst_util_uint64_scale_int (GST_SECOND, v4l_src->fps_out_d, v4l_src->fps_out_n);
	GST_DEBUG (">>V4L_SRC: time per frame %d", (guint32) v4l_src->time_per_frame);
	v4l_src->last_ts = 0;
	return 0;
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_stop_capturing    
        
DESCRIPTION:   This function triggers the V4L Driver to stop Capturing

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_stop_capturing (MFWGstV4LSrc * v4l_src)
{
	enum v4l2_buf_type type;
	guint i;
	gint index;
	gint loop_num =  v4l_src->loop_success;

	if (loop_num == BUFF_ALLOC_OR_FREE)
		loop_num = loop_num + 1;

	while (loop_num >= (BUFF_ALLOC_OR_FREE))
	{
		switch (loop_num)
		{
			case STREAM_ON_OR_OFF:
				GST_DEBUG ("STREAM_OFF \n");
				type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
				if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_STREAMOFF, &type) < 0)
				{
					GST_ERROR (">>V4L_SRC: error in VIDIOC_STREAMOFF");
					return -1;
				}
				v4l_src->start = FALSE;
				break;
			case CREATE_MUTEX_OR_DESTROY:
				GST_DEBUG ("MUTEX_DESTROY \n");
				g_mutex_free (v4l_src->pool_lock);
				v4l_src->pool_lock = NULL;
				break;
			case MAP_BUF_OR_UNMAP:
				GST_DEBUG ("UNMAP_BUF \n");
				for (i = 0; i < v4l_src->num_buf_created; i++)
				{
					GST_DEBUG ("BUFFER LOOP %d \n",i);
					munmap (GST_BUFFER_DATA (v4l_src->buffers[i]),
							GST_BUFFER_SIZE (v4l_src->buffers[i]));
					if (g_list_find (v4l_src->free_pool, (gpointer) (i))) {
						index = G_N_ELEMENTS (v4l_src->buffers[i]->_gst_reserved) - 1;
						gst_buffer_meta_free (v4l_src->buffers[i]->_gst_reserved[index]);
						v4l_src->buffers[i]->_gst_reserved[index] = NULL;
						gst_buffer_unref (v4l_src->buffers[i]);
					}
					else {
						GST_DEBUG ("v4l buffer 0x%x index:%d is being used outside,"
							 "so it will not be un-referred here", v4l_src->buffers[i], i);
					}
				}
				g_list_free (v4l_src->free_pool);
				v4l_src->free_pool = NULL;
				break;
			case BUFF_ALLOC_OR_FREE:
				GST_DEBUG ("BUF_FREE \n");
				if (v4l_src->buffers)
					g_free (v4l_src->buffers);
				break;
			default:
				break;
		}
		loop_num = loop_num - 1;
	}

	return 0;
}

static gint
fgets_with_openclose (gchar * fname, gchar * buf, size_t maxlen)
{
  FILE *fp;
  gchar *s;

  if ((fp = fopen (fname, "r")) != NULL) {
    s = fgets (buf, maxlen, fp);
    fclose (fp);
    return (s != NULL) ? strlen (buf) : 0;
  } else {
    return -1;
  }
}


static gint
open_v4lsubdev (gchar *device_path, const gchar *chip_name)
{
  gint dev_num;
  gint ret = -1;

  gchar subdev_location[MAX_PATH_LENGTH];
  gchar subdev_name[MAX_DEVICE_NAME_LENGTH];
  gchar name_path[MAX_PATH_LENGTH];

  gboolean subdev_found = FALSE;
  gchar *device_node;
  gchar *subdev_dir_name = NULL;

  DIR *main_dev_dir;
  struct dirent *dir;

  device_node = strstr(device_path, "video");

  snprintf (subdev_location, (MAX_PATH_LENGTH - 1),
            "/sys/class/video4linux/%s/device/video4linux/",
            device_node);
  main_dev_dir = opendir(subdev_location);

  if (main_dev_dir) {
    /* Parse through directory names and find v4l subdev directories
       and read the subdev name. When the chip name is contained in
       subdev name, we got the desired subdev*/
    while(   ((dir = readdir(main_dev_dir)) != NULL)
          && (subdev_found == FALSE)) {

      subdev_dir_name = strstr(dir->d_name, "v4l-subdev");
      if (subdev_dir_name) {

        snprintf(name_path, (MAX_PATH_LENGTH - 1),
                 "%s/%s/%s", subdev_location, subdev_dir_name, "name");
        ret = fgets_with_openclose (name_path, subdev_name,
                                    MAX_DEVICE_NAME_LENGTH);

        if (ret > 0) {
          if (strstr (subdev_name, chip_name) != NULL) {
            snprintf (subdev_location, MAX_DEVICE_NAME_LENGTH,
                      "/dev/%s", subdev_dir_name);
	    subdev_found = TRUE;
          }
        }
      }
    }
    closedir(main_dev_dir);

  } else {
      GST_WARNING(">>V4L_SRC: subdev location %s, does not exist",
                  subdev_location);
  }

  if (subdev_found == TRUE) {
    ret = open (subdev_location, O_RDWR, 0);
    if (ret < 0) {
      GST_WARNING(">>V4L_SRC: Failed to open subdevice %s, errno: %d, error: %s",
                  subdev_location, errno, strerror(errno));
    }
  } else {
    GST_WARNING(">>V4L_SRC: subdevice not found for chip: %s, under %s",
                chip_name, subdev_location);
  }
  return ret;
}

/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_capture_setup    
        
DESCRIPTION:   This function does the necessay initialistions for the V4L capture
               device driver.

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_capture_setup (MFWGstV4LSrc * v4l_src)
{
  struct v4l2_format fmt = {0};
  struct v4l2_control ctrl = {0};
  struct v4l2_streamparm parm = {0};
  struct v4l2_crop crop = {0};
  struct v4l2_mxc_offset off;
  gint in_width = 0; 
  gint in_height = 0;
  gint fd_v4l = v4l_src->fd_v4l;
  gint ret;
  gint subdev_fd;

/* Setting boolean capture_start_flag=TRUE, so that color format cannot be 
 * configured once the capture starts
 */
  v4l_src->capture_start_flag = TRUE;

  /**
   * Checking for input flag, based on the flag set in set_property and
   * accordingly set the input param value through VIDIOC_S_INPUT timed_ioctl call
   */
  if (v4l_src->input_flag)
  {
    /* Set input parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_INPUT, &v4l_src->input) < 0)
    {
      GST_ERROR("V4LSRC: Failed to switch the input to %d", v4l_src->input);
    }
    else
    {
      v4l_src->input_flag = FALSE;
    }
  }


  v4l_src->std_id = V4L2_STD_UNKNOWN;
  timed_ioctl (fd_v4l, VIDIOC_G_STD, &v4l_src->std_id);
  GST_INFO ("video standard is %s (0x%llx)\n",
	    v4l_src->std_id == V4L2_STD_NTSC ? "NTSC" :
	    (v4l_src->std_id == V4L2_STD_PAL ? "PAL" :
	     (v4l_src->std_id == V4L2_STD_UNKNOWN ? "unknown" : "other")),
	    v4l_src->std_id);

  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
/* Enhanced to support diff RGB and YUV color-formats */
  #ifdef FSL_COLOR_MODE_SUPPORT
    fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
  #else  
    switch(v4l_src->pix_fmt)
    {
      case PIX_FMT_RGB565:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB565;
        break;

      case PIX_FMT_BGR24:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_BGR24;
        break;
	
      case PIX_FMT_RGB24:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB24;
        break;
	
      case PIX_FMT_BGR32:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_BGR32;
        break;
	
      case PIX_FMT_RGB32:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB32;
        break;
	
      case PIX_FMT_YUV422P:
	      fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV422P;
        break;
	
      case PIX_FMT_UYVY:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_UYVY;
        break;
	
      case PIX_FMT_YUYV:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUYV;
        break;
	
      case PIX_FMT_YUV420: /* YU12 */
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
        break;
	
      case PIX_FMT_NV12:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_NV12;
        break;
	
      default: /* Unlikely, default color format UYVY is set instead */
        v4l_src->pix_fmt = PIX_FMT_UYVY;           
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_UYVY;
        break;	
    }
  #endif


  in_width = v4l_src->capture_width;
  in_height = v4l_src->capture_height;


  fmt.fmt.pix.width = in_width;
  fmt.fmt.pix.height = in_height;

  fmt.fmt.pix.bytesperline = in_width;
  fmt.fmt.pix.priv = 0;
  fmt.fmt.pix.sizeimage = 0;


  parm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  parm.parm.capture.timeperframe.numerator = v4l_src->fps_d;
  parm.parm.capture.timeperframe.denominator = v4l_src->fps_n;

  if (timed_ioctl (fd_v4l, VIDIOC_S_PARM, &parm) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_S_PARM failed");
    return 0;
  }

  subdev_fd = open_v4lsubdev (v4l_src->devicename, DEINTERLACE_DEVICE_NAME);

  if (subdev_fd >= 0) {
    struct v4l2_subdev_frame_interval output_f_interval;
    memset(&output_f_interval, 0, sizeof(output_f_interval));

    output_f_interval.interval.numerator = v4l_src->fps_out_d;
    output_f_interval.interval.denominator = v4l_src->fps_out_n;

    ret = ioctl(subdev_fd, VIDIOC_SUBDEV_S_FRAME_INTERVAL, &output_f_interval);

    if (ret < 0) {
      log_errmem("vdic", "MFW_V4lSRC ioctl %s is failed, with error: %d\n",
                 "VIDIOC_SUBDEV_S_FRAME_INTERVAL", ret);
      GST_ERROR (">>V4L_SRC: ioctl %s is failed, with error: %d, %d",
                 "VIDIOC_SUBDEV_S_FRAME_INTERVAL", ret, errno);

    }
    close (subdev_fd);
  } else {
    GST_WARNING(">>V4L_SRC: Unable to set output frame rate for de-interlacer");
  }

  if (timed_ioctl (fd_v4l, VIDIOC_S_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: set format failed");
    return 0;
  }

  if(v4l_src->crop_width && v4l_src->crop_height)
  {
      crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      crop.c.left = v4l_src->crop_left;
      crop.c.top = v4l_src->crop_top;
      crop.c.width = v4l_src->crop_width;
      crop.c.height = v4l_src->crop_height;
      GST_LOG("set cropping: %dx%d -> %dx%d", v4l_src->crop_left,
              v4l_src->crop_top, v4l_src->crop_width,v4l_src->crop_height);

      if (timed_ioctl(fd_v4l, VIDIOC_S_CROP, &crop) < 0)
      {
          GST_ERROR("set cropping failed\n");
          return 0;
      }

      if (timed_ioctl(fd_v4l, VIDIOC_G_CROP, &crop) < 0)
      {
          GST_ERROR("get cropping failed\n");
          return 0;
      }

      GST_LOG("cropping was set to: %dx%d -> %dx%d",
           crop.c.left, crop.c.top,
           crop.c.width, crop.c.height);
  }

  if (timed_ioctl (fd_v4l, VIDIOC_G_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return 0;
  }



  // Set rotation and horizontal/vertical flips
  ctrl.id = V4L2_CID_ROTATE;
  ctrl.value = v4l_src->rotate;
  if (timed_ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: rotation set ctrl failed");
    return 0;
  }

  ctrl.id = V4L2_CID_HFLIP;
  ctrl.value = (v4l_src->hflip == TRUE) ? 1 : 0;
  if (timed_ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: hflip set ctrl failed");
    return 0;
  }

  ctrl.id = V4L2_CID_VFLIP;
  ctrl.value = (v4l_src->vflip == TRUE) ? 1 : 0;
  if (timed_ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: vflip set ctrl failed");
    return 0;
  }

  /**
   * Checking if any picture enhancement parameter needs to be set based on the
   * flag set in set_property and set the value through timed_ioctl call
   */
  if (v4l_src->brightness_set_flag)
  {
    ctrl.id = V4L2_CID_BRIGHTNESS;
    ctrl.value = v4l_src->brightness;

    /* Set brightness parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR(">>V4LSRC: set brightness control failed");
    }
    else
    {
      v4l_src->brightness_set_flag = FALSE;
    }
  }

  if (v4l_src->contrast_set_flag)
  {
    ctrl.id = V4L2_CID_CONTRAST;
    ctrl.value = v4l_src->contrast;

    /* Set contrast parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR(">>V4LSRC: set contrast control failed");
    }
    else
    {
      v4l_src->contrast_set_flag = FALSE;
    }
  }

  if (v4l_src->saturation_set_flag)
  {
    ctrl.id = V4L2_CID_SATURATION;
    ctrl.value = v4l_src->saturation;

    /* Set saturation parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR("V4LSRC: set saturtion control failed");
    }
    else
    {
      v4l_src->saturation_set_flag = FALSE;
    }
  }

  if (v4l_src->hue_set_flag)
  {
    ctrl.id = V4L2_CID_HUE;
    ctrl.value = v4l_src->hue;

    /* Set hue parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR("V4LSRC: set hue control failed");
    }
    else
    {
      v4l_src->hue_set_flag = FALSE;
    }
  }

  if (v4l_src->motion_set_flag)
  {
    ctrl.id = V4L2_CID_IMX6_MOTION;
    ctrl.value = v4l_src->motion;

    /* Set deinterlace motion compensation parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR("V4LSRC: set deinterlace motion control failed");
    }
    else
    {
      v4l_src->motion_set_flag = FALSE;
    }
  }

  /**
   * Checking if any chroma parameter needs to be set based on the
   * flag set in set_property and set the value through timed_ioctl call
   */
  if (v4l_src->cb_set_flag)
  {
    ctrl.id = V4L2_CID_ADV718x_OFF_CB;
    ctrl.value = v4l_src->chroma_blue;

    /* Set chroma blue parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR("V4LSRC: set chroma blue parameter failed");
    }
    else
    {
        v4l_src->cb_set_flag = FALSE;
    }
  }

  if (v4l_src->cr_set_flag)
  {
    ctrl.id = V4L2_CID_ADV718x_OFF_CR;
    ctrl.value = v4l_src->chroma_red;

    /* Set chroma red parameter */
    if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
    {
      GST_ERROR("V4LSRC: set chroma red parameter failed");
    }
    else
    {
        v4l_src->cr_set_flag = FALSE;
    }
  }

  if (fmt.fmt.pix.width != v4l_src->capture_width ||
      fmt.fmt.pix.height != v4l_src->capture_height) {
	  GST_WARNING ("capture format changed to %dx%d",
		       fmt.fmt.pix.width, fmt.fmt.pix.height);
	  v4l_src->capture_width = fmt.fmt.pix.width;
	  v4l_src->capture_height = fmt.fmt.pix.height;
  }


  struct v4l2_requestbuffers req;
  memset (&req, 0, sizeof (req));
  req.count = v4l_src->queue_size;
  req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  req.memory = V4L2_MEMORY_MMAP;

  if (timed_ioctl (fd_v4l, VIDIOC_REQBUFS, &req) < 0) {
    GST_ERROR
        (">>V4L_SRC: v4l_mfw_gst_v4lsrc_capture_setup: VIDIOC_REQBUFS failed");
    log_errmem
        (v4l_src->devicename, "MFW_V4LSRC: v4l_mfw_gst_v4lsrc_capture_setup: VIDIOC_REQBUFS failed\n");
    return 0;
  }

  return TRUE;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_set_property   
        
DESCRIPTION:        This function is notified if application changes the values of 
                    a property.            

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (object);
  struct v4l2_control ctrl;
  gint fd_v4l = v4l_src->fd_v4l;

  /* mutex lock, so that device file cannot be closed when setting the values */
  g_mutex_lock(&v4l_src->mutex);

  switch (prop_id) {
    case MFW_V4L_SRC_WIDTH:
      v4l_src->capture_width = g_value_get_int (value);
      GST_DEBUG ("width=%d", v4l_src->capture_width);
      break;
    case MFW_V4L_SRC_HEIGHT:
      v4l_src->capture_height = g_value_get_int (value);
      GST_DEBUG ("height=%d", v4l_src->capture_height);
      break;
    case MFW_V4L_SRC_ROTATE:
      v4l_src->rotate = g_value_get_int (value);
      GST_DEBUG ("rotate=%d", v4l_src->rotate);
      break;
    case MFW_V4L_SRC_HFLIP:
      v4l_src->hflip = g_value_get_boolean (value);
      GST_DEBUG ("hflip=%d", v4l_src->hflip);
      break;
    case MFW_V4L_SRC_VFLIP:
      v4l_src->vflip = g_value_get_boolean (value);
      GST_DEBUG ("vflip=%d", v4l_src->vflip);
      break;

    case MFW_V4L_SRC_PREVIEW:
      v4l_src->preview = g_value_get_boolean (value);
      GST_DEBUG ("preview=%d", v4l_src->preview);
      break;


    case MFW_V4L_SRC_PREVIEW_WIDTH:
      v4l_src->preview_width = g_value_get_int (value);
      GST_DEBUG ("preview_width=%d", v4l_src->preview_width);
      break;

    case MFW_V4L_SRC_PREVIEW_HEIGHT:
      v4l_src->preview_height = g_value_get_int (value);
      GST_DEBUG ("preview_height=%d", v4l_src->preview_height);
      break;

    case MFW_V4L_SRC_PREVIEW_TOP:
      v4l_src->preview_top = g_value_get_int (value);
      GST_DEBUG ("preview_top=%d", v4l_src->preview_top);
      break;

    case MFW_V4L_SRC_PREVIEW_LEFT:
      v4l_src->preview_left = g_value_get_int (value);
      GST_DEBUG ("preview_left=%d", v4l_src->preview_left);
      break;
    case MFW_V4L_SRC_PREVIEW_COLOR_KEY:
      v4l_src->preview_color_key = g_value_get_uint (value);
      v4l_src->preview_color_key_en = TRUE;
      GST_DEBUG ("preview_color_key = 0x%08x", v4l_src->preview_color_key);
      break;
    case MFW_V4L_SRC_PREVIEW_GALPHA:
      v4l_src->preview_galpha = g_value_get_int (value);
      GST_DEBUG ("preview_galpha = %d", v4l_src->preview_galpha);
      break;

    case MFW_V4L_SRC_CROP_LEFT:
      v4l_src->crop_left = g_value_get_int (value);
      GST_DEBUG ("crop_left=%d", v4l_src->crop_left);
      break;
    case MFW_V4L_SRC_CROP_TOP:
      v4l_src->crop_top = g_value_get_int (value);
      GST_DEBUG ("crop_top=%d", v4l_src->crop_top);
      break;
    case MFW_V4L_SRC_CROP_WIDTH:
      v4l_src->crop_width = g_value_get_int (value);
      GST_DEBUG ("crop_width=%d", v4l_src->crop_width);
      break;
    case MFW_V4L_SRC_CROP_HEIGHT:
      v4l_src->crop_height = g_value_get_int (value);
      GST_DEBUG ("crop_height=%d", v4l_src->crop_height);
      break;
    case MFW_V4L_SRC_FRAMERATE_NUM:
      v4l_src->fps_n = g_value_get_int (value);
      GST_DEBUG ("framerate numerator =%d", v4l_src->fps_n);
      break;
    case MFW_V4L_SRC_FRAMERATE_DEN:
      v4l_src->fps_d = g_value_get_int (value);
      GST_DEBUG ("framerate denominator=%d", v4l_src->fps_d);
      break;
    case MFW_V4L_SRC_OUTPUT_FRAMERATE_NUM:
      v4l_src->fps_out_n = g_value_get_int (value);
      GST_DEBUG ("output framerate numerator=%d", v4l_src->fps_out_n);
      break;
    case MFW_V4L_SRC_OUTPUT_FRAMERATE_DEN:
      v4l_src->fps_out_d = g_value_get_int (value);
      GST_DEBUG ("output framerate denominator=%d", v4l_src->fps_out_d);
      break;

    case MFW_V4L_SRC_INPUT:
      v4l_src->input  = g_value_get_int (value);
      GST_DEBUG ("input=%d", v4l_src->input);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->input_flag = TRUE;
      }
      else
      {
        /* this ioctl sets input parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_INPUT, &v4l_src->input) < 0)
          GST_ERROR("V4LSRC: set input parameter failed");
      }
      break;

    case MFW_V4L_SRC_DEVICE:
      if (v4l_src->devicename)
        g_free (v4l_src->devicename);
      v4l_src->devicename = g_strdup (g_value_get_string (value));
      break;

    case MFW_V4L_SRC_QUEUE_SIZE:
      v4l_src->queue_size = g_value_get_int (value);
      GST_DEBUG ("queue size=%d", v4l_src->queue_size);
      break;

/* Enhanced to support diff RGB and YUV color-formats */
    case MFW_V4L_SRC_PIXEL_FMT:
      if (TRUE != v4l_src->capture_start_flag)
      {
        v4l_src->pix_fmt = g_value_get_uint(value);
        GST_DEBUG("Pixel format=%u", v4l_src->pix_fmt);
      }
      else
      {
        GST_ERROR("Capturing started, cannot set the color format \n");      
      }                           
      break;
      
/* Enhanced to support picture enhancement parameters */
    case MFW_V4L_SRC_BRIGHTNESS:
      v4l_src->brightness = g_value_get_int(value);
      GST_DEBUG("Brightness parameter=%d", v4l_src->brightness);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->brightness_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_BRIGHTNESS;
        ctrl.value = v4l_src->brightness;

        /* this ioctl sets brightness parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set brightness control failed");
      }
      break;

    case MFW_V4L_SRC_CONTRAST:
      v4l_src->contrast = g_value_get_int(value);
      GST_DEBUG("Contrast parameter=%d", v4l_src->contrast);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->contrast_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_CONTRAST;
        ctrl.value = v4l_src->contrast;

        /* this ioctl sets contrast parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set contrast control failed");
      }
      break;

    case MFW_V4L_SRC_SATURATION:
      v4l_src->saturation = g_value_get_int(value);
      GST_DEBUG("Saturation parameter=%d", v4l_src->saturation);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->saturation_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_SATURATION;
        ctrl.value = v4l_src->saturation;

        /* this ioctl sets saturation parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set saturtion control failed");
      }
      break;

    case MFW_V4L_SRC_HUE:
      v4l_src->hue = g_value_get_int(value);
      GST_DEBUG("Hue parameter=%d", v4l_src->hue);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->hue_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_HUE;
        ctrl.value = v4l_src->hue;

        /* this ioctl sets hue parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set hue control failed");
      }
      break;

/* Enhanced to set deinterlace motion compensation parameters */
    case MFW_V4L_SRC_MOTION:
      v4l_src->motion = g_value_get_int(value);
      GST_DEBUG("Motion parameter=%d", v4l_src->motion);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->motion_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_IMX6_MOTION;
        ctrl.value = v4l_src->motion;

        /* this ioctl sets deinterlace motion compensation parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set motion control failed");
      }
      break;

/* Enhanced to support chroma components */
    case MFW_V4L_SRC_CHROMA_BLUE:
      v4l_src->chroma_blue = g_value_get_int(value);
      GST_DEBUG("Chroma Blue=%d", v4l_src->chroma_blue);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->cb_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_ADV718x_OFF_CB;
        ctrl.value = v4l_src->chroma_blue;

        /* this ioctl sets chroma blue parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set chroma blue parameter failed");
      }
      break;

    case MFW_V4L_SRC_CHROMA_RED:
	  v4l_src->chroma_red = g_value_get_int(value);
	  GST_DEBUG("Chroma Red=%d", v4l_src->chroma_red);

	  if (fd_v4l < 0)
	  {
		GST_INFO("V4LSRC: fd_v4l is not opened");
		v4l_src->cr_set_flag = TRUE;
	  }
	  else
	  {
		ctrl.id = V4L2_CID_ADV718x_OFF_CR;
		ctrl.value = v4l_src->chroma_red;

		/* this ioctl sets chroma red parameter */
		if (timed_ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
		  GST_ERROR("V4LSRC: set chroma red parameter failed");
	  }
      break;

    case MFW_V4L_STREAM_GOOD_BUFFER:
      v4l_src->stream_good_buf_only = g_value_get_boolean(value);
      GST_DEBUG("stream-good-buf-only=%d", v4l_src->stream_good_buf_only);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_mutex_unlock(&v4l_src->mutex);
}


/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_get_property    
        
DESCRIPTION:    This function is notified if application requests the values of 
                a property.                  

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{

  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (object);
  struct v4l2_control ctrl;
  gint fd_v4l = v4l_src->fd_v4l;

  /* mutex lock, so that device file cannot be closed when getting the values */
  g_mutex_lock(&v4l_src->mutex);

  switch (prop_id) {
    case MFW_V4L_SRC_WIDTH:
      g_value_set_int (value, v4l_src->capture_width);
      break;
    case MFW_V4L_SRC_HEIGHT:
      g_value_set_int (value, v4l_src->capture_height);
      break;
    case MFW_V4L_SRC_ROTATE:
      g_value_set_int (value, v4l_src->rotate);
      break;
    case MFW_V4L_SRC_HFLIP:
      g_value_set_boolean (value, v4l_src->hflip);
      break;
    case MFW_V4L_SRC_VFLIP:
      g_value_set_boolean (value, v4l_src->vflip);
      break;

    case MFW_V4L_SRC_PREVIEW:
      g_value_set_boolean (value, v4l_src->preview);
      break;
    case MFW_V4L_SRC_PREVIEW_WIDTH:
      g_value_set_int (value, v4l_src->preview_width);
      break;
    case MFW_V4L_SRC_PREVIEW_HEIGHT:
      g_value_set_int (value, v4l_src->preview_height);
      break;

    case MFW_V4L_SRC_PREVIEW_TOP:
      g_value_set_int (value, v4l_src->preview_top);
      break;

    case MFW_V4L_SRC_PREVIEW_LEFT:
      g_value_set_int (value, v4l_src->preview_left);
      break;

    case MFW_V4L_SRC_PREVIEW_COLOR_KEY:
      g_value_set_uint (value, v4l_src->preview_color_key);
      break;
    case MFW_V4L_SRC_PREVIEW_GALPHA:
      g_value_set_int (value, v4l_src->preview_galpha);
      break;

    case MFW_V4L_SRC_CROP_LEFT:
      g_value_set_int (value, v4l_src->crop_left);
      break;
    case MFW_V4L_SRC_CROP_TOP:
      g_value_set_int (value, v4l_src->crop_top);
      break;
    case MFW_V4L_SRC_CROP_WIDTH:
      g_value_set_int (value, v4l_src->crop_width);
      break;
    case MFW_V4L_SRC_CROP_HEIGHT:
      g_value_set_int (value, v4l_src->crop_height);
      break;
    case MFW_V4L_SRC_FRAMERATE_NUM:
      g_value_set_int (value, v4l_src->fps_n);
      break;
    case MFW_V4L_SRC_FRAMERATE_DEN:
      g_value_set_int (value, v4l_src->fps_d);
      break;
    case MFW_V4L_SRC_OUTPUT_FRAMERATE_NUM:
      g_value_set_int (value, v4l_src->fps_out_n);
      break;
    case MFW_V4L_SRC_OUTPUT_FRAMERATE_DEN:
      g_value_set_int (value, v4l_src->fps_out_d);
      break;


    case MFW_V4L_SRC_INPUT:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened \n");
      else
      {
        /* this ioctl gets input parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_INPUT, &v4l_src->input) < 0)
          GST_ERROR("V4LSRC: get input parameter failed");
        else
        {
          GST_DEBUG("Input retreived=%d", v4l_src->input);
        }
      }

      g_value_set_int (value, v4l_src->input);
      break;

    case MFW_V4L_SRC_DEVICE:
      g_value_set_string (value, v4l_src->devicename);
      break;

    case MFW_V4L_SRC_QUEUE_SIZE:
      g_value_set_int (value, v4l_src->queue_size);
      break;

/* Enhanced to support diff RGB and YUV color-formats */
    case MFW_V4L_SRC_PIXEL_FMT:
      g_value_set_uint(value, v4l_src->pix_fmt);
      break;
      
/* Enhanced to support picture enhancement parameters */
    case MFW_V4L_SRC_BRIGHTNESS:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened \n");
      else
      {
        ctrl.id = V4L2_CID_BRIGHTNESS;

        /* this ioctl gets brightness parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get brightness control failed");
        else
        {
          GST_DEBUG("Brightness value=%d", ctrl.value);
          v4l_src->brightness = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->brightness);
      break;

    case MFW_V4L_SRC_CONTRAST:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_CONTRAST;

        /* this ioctl gets contrast parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get contrast control failed");
        else
        {
          GST_DEBUG("Contrast value=%d", ctrl.value);
          v4l_src->contrast = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->contrast);
      break;

    case MFW_V4L_SRC_SATURATION:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_SATURATION;

        /* this ioctl gets saturation parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get saturation control failed");
        else
        {
          GST_DEBUG("Saturation value=%d", ctrl.value);
          v4l_src->saturation = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->saturation);
      break;

    case MFW_V4L_SRC_HUE:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_HUE;

        /* this ioctl gets hue parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get hue control failed");
        else
        {
          GST_DEBUG("Hue value=%d", ctrl.value);
          v4l_src->hue = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->hue);
      break;

/* Enhanced to set deinterlace motion compensation parameters */
    case MFW_V4L_SRC_MOTION:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_IMX6_MOTION;

        /* this ioctl gets deinterlace motion compensation parameter */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get motion control failed");
        else
        {
          GST_DEBUG("Motion value=%d", ctrl.value);
          v4l_src->motion = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->motion);
      break;

/* Enhanced to support chroma components */
    case MFW_V4L_SRC_CHROMA_BLUE:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_ADV718x_OFF_CB;

        /* this ioctl gets chroma blue parameter value */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get chroma blue parameter value failed");
        else
        {
          GST_DEBUG("Chroma Blue value=%d", ctrl.value);
          v4l_src->chroma_blue = ctrl.value;
        }
      }
      break;

    case MFW_V4L_SRC_CHROMA_RED:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_ADV718x_OFF_CR;

        /* this ioctl gets chroma red parameter value */
        if (timed_ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get chroma red parameter value failed");
        else
        {
          GST_DEBUG("Chroma Red value=%d", ctrl.value);
          v4l_src->chroma_red = ctrl.value;
        }
      }
      break;
    case MFW_V4L_STREAM_GOOD_BUFFER:
      g_value_set_boolean(value, v4l_src->stream_good_buf_only);
    break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_mutex_unlock(&v4l_src->mutex);
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_set_caps
         
DESCRIPTION:         this function does the capability negotiation between adjacent pad

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        caps      -   pointer to GstCaps
        
  
RETURN VALUE:       TRUE or FALSE depending on capability is negotiated or not.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_set_caps (GstBaseSrc * src, GstCaps * caps)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);

  return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_overlay_setup
         
DESCRIPTION:         This function performs the initialisations required for preview

ARGUMENTS PASSED:    
        fd_v4l    -   capture device ID
        fmt       -   pointer to the V4L format structure.
        
  
RETURN VALUE:       TRUE - preview setup initialised successfully
                    FALSE - Error in initialising the preview set up.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
mfw_gst_v4lsrc_overlay_setup (MFWGstV4LSrc * v4l_src, struct v4l2_format * fmt)
{
  struct v4l2_streamparm parm;
  struct v4l2_control ctl;
  struct v4l2_crop crop = {0};
  int g_sensor_top = 0;
  int g_sensor_left = 0;
  int g_camera_color = 0;
  int fd_v4l = v4l_src->fd_v4l;

  GST_INFO ("display lcd:%d\n", v4l_src->g_display_lcd);
  /* this ioctl sets up the LCD display for preview */
  if (timed_ioctl (fd_v4l, VIDIOC_S_OUTPUT, &v4l_src->g_display_lcd) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_S_OUTPUT failed");
    return FALSE;
  }

  ctl.id = V4L2_CID_PRIVATE_BASE + 2;
  ctl.value = v4l_src->rotate;

  /* this ioctl sets rotation value on the display */
  if (timed_ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl) < 0) {
    GST_ERROR (">>V4L_SRC: rotation set control failed");
    return FALSE;
  }

  if (timed_ioctl (fd_v4l, VIDIOC_S_FMT, fmt) < 0) {
    GST_ERROR (">>V4L_SRC: set format failed");
    return FALSE;
  }

  if(v4l_src->crop_width && v4l_src->crop_height)
  {
      crop.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
      crop.c.left = v4l_src->crop_left;
      crop.c.top = v4l_src->crop_top;
      crop.c.width = v4l_src->crop_width;
      crop.c.height = v4l_src->crop_height;

      /* this timed_ioctl sets capture rectangle */
      if (timed_ioctl(fd_v4l, VIDIOC_S_CROP, &crop) < 0)
      {
        GST_ERROR(">>V4L_SRC: set capture rectangle for cropping failed");
        return FALSE;
      }
  }

  if (timed_ioctl (fd_v4l, VIDIOC_G_FMT, fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return FALSE;
  }

  return TRUE;
}



/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_start_preview
         
DESCRIPTION:         This function starts the preview of capture

ARGUMENTS PASSED:    
        fd_v4l    -   capture device ID
        
  
RETURN VALUE:        TRUE - preview start initialised successfully
                     FALSE - Error in starting the preview
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
mfw_gst_v4lsrc_start_preview (int fd_v4l)
{
  int i;
  int overlay = 1;
  struct v4l2_control ctl;
  int g_camera_color = 0;

  if (ioctl (fd_v4l, VIDIOC_OVERLAY, &overlay) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_OVERLAY start failed");
    return FALSE;
  }

  for (i = 0; i < 3; i++) {
    // flash a frame
    ctl.id = V4L2_CID_PRIVATE_BASE + 1;
    if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl) < 0) {
      GST_ERROR (">>V4L_SRC: set ctl failed");
      return FALSE;
    }
    sleep (1);
  }

  return TRUE;
}

static gboolean
mfw_gst_v4lsrc_setup_preview (MFWGstV4LSrc * v4l_src)
{
  struct v4l2_crop crop;
  struct v4l2_framebuffer fbuf;
  struct v4l2_format fmt;

  /* set overlay framebuffer pixel format via S_FBUF */
  memset (&fbuf, 0, sizeof (fbuf));
  fbuf.fmt.pixelformat = V4L2_PIX_FMT_RGB565;
  if (v4l_src->preview_galpha != 0)
    fbuf.flags |= V4L2_FBUF_FLAG_GLOBAL_ALPHA;
  if (v4l_src->preview_color_key_en == TRUE)
    fbuf.flags |= V4L2_FBUF_FLAG_SRC_CHROMAKEY;

  if (timed_ioctl(v4l_src->fd_v4l, VIDIOC_S_FBUF, &fbuf) < 0) {
    GST_ERROR("VIDIOC_S_FBUF failed\n");
    return FALSE;
  }

  fmt.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
  fmt.fmt.win.w.top = v4l_src->preview_top;
  fmt.fmt.win.w.left = v4l_src->preview_left;
  fmt.fmt.win.w.width = v4l_src->preview_width;
  fmt.fmt.win.w.height = v4l_src->preview_height;
  fmt.fmt.win.chromakey = v4l_src->preview_color_key;
  fmt.fmt.win.global_alpha = v4l_src->preview_galpha;

  /* this function sets up the V4L for preview */
  if (mfw_gst_v4lsrc_overlay_setup (v4l_src, &fmt) == FALSE) {
    GST_ERROR (">>V4L_SRC: Setup overlay failed.");
    return FALSE;
  }


  return TRUE;
}

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_start
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to start the video capturing process 
                     from this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
RETURN VALUE:        TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_start (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  struct v4l2_format fmt;

  if (!mfw_gst_v4lsrc_capture_setup (v4l_src)) {
    GST_ERROR ("capture_setup failed");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {
    mfw_gst_v4lsrc_setup_preview(v4l_src);
  }

  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_G_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return FALSE;
  } else {
    v4l_src->buffer_size = fmt.fmt.pix.sizeimage;
    GST_DEBUG ("Width = %d", fmt.fmt.pix.width);
    GST_DEBUG ("Height = %d", fmt.fmt.pix.height);
    GST_DEBUG ("Image size = %d", fmt.fmt.pix.sizeimage);
    GST_DEBUG ("pixelformat = %d", fmt.fmt.pix.pixelformat);
  }

  if (mfw_gst_v4lsrc_start_capturing (v4l_src) < 0) {
    GST_ERROR ("start_capturing failed");
    if (mfw_gst_v4lsrc_stop_capturing (v4l_src) < 0)
      GST_ERROR (">>V4L_SRC: ERROR in freeing allocated resource");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {
    mfw_gst_v4lsrc_start_preview (v4l_src->fd_v4l);
  }
  v4l_src->fist_frame_after_start = TRUE;
  v4l_src->offset = 0;
  return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_stop
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to stop the video capturing process 
                     by this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
  
RETURN VALUE:        TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_stop (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  gint overlay = 0;

  if (mfw_gst_v4lsrc_stop_capturing (v4l_src) < 0) {
    GST_ERROR (">>V4L_SRC: stop_capturing failed");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {

    if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_OVERLAY, &overlay) < 0) {
      printf ("VIDIOC_OVERLAY stop failed\n");
      return FALSE;
    }
  }

/* Setting boolean capture_start_flag=FALSE, so that color format can be 
 * configured 
 */
  v4l_src->capture_start_flag = FALSE;
  v4l_src->fist_frame_after_start = FALSE;
  return TRUE;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_new
         
DESCRIPTION:        This function is used to store the frames captured by the
                    V4L capture driver

ARGUMENTS PASSED:   v4l_src     - 
        
RETURN VALUE:       TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstBuffer *
mfw_gst_v4lsrc_get_buf(MFWGstV4LSrc * v4l_src, guint *buf_flags)
{
  GstBuffer *buf;
  gint fps_n, fps_d;
  struct v4l2_buffer v4lbuf;
  GstClockTime ts, res;
  gint wait_cnt = 0;
  gboolean notify_status = FALSE;

  v4l_src->count++;
  memset (&v4lbuf, 0, sizeof (v4lbuf));
  v4lbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  v4lbuf.memory = V4L2_MEMORY_MMAP;

  while (g_list_length (v4l_src->free_pool) == 0) {
      GST_WARNING ("no buffer available in pool");
      g_usleep(NO_BUFFER_WAIT_TIME_MS*1000);
      wait_cnt ++;
      if (wait_cnt >= NO_BUFFER_WAIT_COUNT) {
          GST_ERROR ("no buffer available in %d ms", NO_BUFFER_WAIT_TIME_MS*NO_BUFFER_WAIT_COUNT);
          return NULL;
      }
  }

  if (v4l_src->stop == TRUE) {
          GST_WARNING("v4l src stopped.");
          return NULL;
  }

  if (timed_ioctl (v4l_src->fd_v4l, VIDIOC_DQBUF, &v4lbuf) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_DQBUF failed.");
    return NULL;
  }

  *buf_flags = v4lbuf.flags;

  if (v4l_src->fist_frame_after_start)
  {
    v4l_src->fist_frame_after_start = FALSE;
    notify_status = TRUE;
  }
  else if (  ((V4L2_BUF_FLAG_ERROR == (v4lbuf.flags & V4L2_BUF_FLAG_ERROR)) &&
        (TRUE != v4l_src->v4l2_buf_flag_err))
     || (!(V4L2_BUF_FLAG_ERROR == (v4lbuf.flags & V4L2_BUF_FLAG_ERROR)) &&
          (TRUE == v4l_src->v4l2_buf_flag_err)))
  {
    notify_status = TRUE;
  }
  else
  {
    /* Nothing to be done */
  }

  if (TRUE == notify_status)
  {
    /**
     * Add check for V4L2_BUF_FLAG_ERROR flag, if set
     * need to discard the captured data buffer and signal is
     * emitted to notify the gstreamer framework captured frame
     * data is error
     */
    if (V4L2_BUF_FLAG_ERROR == (v4lbuf.flags & V4L2_BUF_FLAG_ERROR))
    {
      v4l_src->v4l2_buf_flag_err = TRUE;
      g_signal_emit(v4l_src, gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_ERROR], 0,
                  GST_BASE_SRC_PAD(v4l_src));
      GST_DEBUG ("notified invalid signal \n");
    }
    /**
     * SIGNAL_V4L2_BUF_FLAG_OK signal is emitted to notify the gstreamer
     * framework that captured frame data is correct, if v4l2_buf_flag_err is set
     * to TRUE and if V4L2_BUF_FLAG_ERROR buffer flag is not set
     */
    else
    {
      v4l_src->v4l2_buf_flag_err = FALSE;
      g_signal_emit(v4l_src, gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_OK], 0,
                  GST_BASE_SRC_PAD(v4l_src));
      GST_DEBUG ("notified valid signal \n");
    }
  }

  g_mutex_lock (v4l_src->pool_lock);
  if (g_list_find (v4l_src->free_pool, (gpointer) (v4lbuf.index)))
    GST_LOG ("v4l buffer index:%d will be used outside", v4lbuf.index);
  else
    GST_WARNING ("v4l buffer index:%d can not be found in pool", v4lbuf.index);

  v4l_src->free_pool =
      g_list_remove (v4l_src->free_pool, (gpointer) (v4lbuf.index));

  g_mutex_unlock (v4l_src->pool_lock);

  buf = (GstBuffer *) (v4l_src->buffers[v4lbuf.index]);

  /*
   * If captured frame data is error, pass appropriate GstBuffer
   * flag information as part of v4lsrc GstBuffer
   */
  if (v4l_src->v4l2_buf_flag_err)
  {
    GST_BUFFER_FLAG_UNSET (buf, GST_BUFFER_FLAG_LAST);
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_CORRUPTED);
  }
  else
  {
    GST_BUFFER_FLAG_UNSET (buf, GST_BUFFER_FLAG_CORRUPTED);
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_LAST);
  }

  GST_LOG ("v4l dequeued buffer index:%d(ref %d), num in pool:%d", v4lbuf.index,
      buf->mini_object.refcount, g_list_length (v4l_src->free_pool));

  GST_BUFFER_SIZE (buf) = v4l_src->buffer_size;

  ts = gst_clock_get_time (GST_ELEMENT (v4l_src)->clock);
  if (ts != GST_CLOCK_TIME_NONE)
    ts -= gst_element_get_base_time (GST_ELEMENT (v4l_src));
  else
    ts = v4l_src->count * v4l_src->time_per_frame;
  GST_BUFFER_TIMESTAMP (buf) = ts;
  GST_BUFFER_DURATION (buf) = v4l_src->time_per_frame;

  if (v4l_src->last_ts) {
    guint num_frame_delay = 0;
    GstClockTimeDiff diff = ts - v4l_src->last_ts;
    if (ts < v4l_src->last_ts)
      diff = v4l_src->last_ts + ts;
    while (diff > v4l_src->time_per_frame) {
      diff -= v4l_src->time_per_frame;
      num_frame_delay++;
    }
    if (num_frame_delay > 1)
      GST_LOG (">>V4L_SRC: Camera ts late by %d frames", num_frame_delay);
  }
  v4l_src->last_ts = ts;

  gst_buffer_set_caps (buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (v4l_src)));
  return buf;
}

static GstBuffer *
mfw_gst_v4lsrc_buffer_new (MFWGstV4LSrc * v4l_src)
{
  GstBuffer *buf = NULL;
  guint buf_flags = 0;
  gboolean wait_for_good_buf = FALSE;

  do {
    wait_for_good_buf = FALSE;
    buf = mfw_gst_v4lsrc_get_buf(v4l_src, &buf_flags);
    if (   (TRUE == v4l_src->stream_good_buf_only) && (NULL != buf)
           && ((buf_flags & V4L2_BUF_FLAG_ERROR) == V4L2_BUF_FLAG_ERROR)) {
      /*Give buffer back to pool*/
      gst_buffer_unref(buf);
      wait_for_good_buf = TRUE;
    }
  } while (TRUE == wait_for_good_buf);

  return buf;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_create
         
DESCRIPTION:         This function is registered with the Base Source Class 
                     This function updates the the buffer to be pushed to the
                     next element with the frame captured.
                     
ARGUMENTS PASSED:    v4l_src     - 
        
  
RETURN VALUE:        
              GST_FLOW_OK       -    buffer create successfull.
              GST_FLOW_ERROR    -    Error in buffer creation.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
mfw_gst_v4lsrc_create (GstPushSrc * src, GstBuffer ** buf)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  *buf = mfw_gst_v4lsrc_buffer_new (v4l_src);
  if (*buf == NULL)
    return GST_FLOW_ERROR;
  else
    return GST_FLOW_OK;

}

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_unlock
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to unlock any block in mfw_gst_v4lsrc_create
                     by this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
  
RETURN VALUE:        TRUE
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_unlock (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  v4l_src->stop = TRUE;

  return TRUE;
}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_change_state

DESCRIPTION: this function keeps track of different states of pipeline.

ARGUMENTS PASSED:
        element     -   pointer to element
        transition  -   state of the pipeline

RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_v4lsrc_change_state(GstElement* element, GstStateChange transition)
{
    
    GstStateChangeReturn retstate = GST_STATE_CHANGE_FAILURE;
    MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (element);
    gint fd_v4l = 0;
    int saved_errno = 0;
       
    switch (transition)
    {
        case GST_STATE_CHANGE_NULL_TO_READY:
            GST_DEBUG(" in NULL to READY state \n");
            g_mutex_lock(&v4l_src->mutex);
            fd_v4l = open (v4l_src->devicename, O_RDWR, 0);
            saved_errno = errno;
            g_mutex_unlock(&v4l_src->mutex);

            if (fd_v4l < 0) {
              GST_ERROR (">>V4L_SRC: Unable to open %s", v4l_src->devicename);
              log_errmem(v4l_src->devicename,"MFW_V4LSRC: Failed to open device %s: %s", v4l_src->devicename,
                         strerror(saved_errno));
              return retstate;
            }
            v4l_src->fd_v4l = fd_v4l;
            break;
        case GST_STATE_CHANGE_READY_TO_PAUSED:
            GST_DEBUG(" in READY_TO_PAUSED state \n");
            break;
        case  GST_STATE_CHANGE_PAUSED_TO_PLAYING:
            GST_DEBUG(" in  to  PAUSED_TO_PLAYING state \n");
            v4l_src->stop = FALSE;
            break;
        default:
            break;
    }

    retstate = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

    switch (transition)
    {
        case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
            GST_DEBUG(" in  to PLAYING_TO_PAUSED state \n");
            v4l_src->stop = TRUE;
            break;
        case GST_STATE_CHANGE_PAUSED_TO_READY:
            GST_DEBUG(" in  to PAUSED_TO_READY state \n");
            break;
        case GST_STATE_CHANGE_READY_TO_NULL:
            GST_DEBUG(" in  to READY_TO_NULL state \n");
            g_mutex_lock(&v4l_src->mutex);
            close (v4l_src->fd_v4l);
            v4l_src->fd_v4l = -1;
            g_mutex_unlock(&v4l_src->mutex);
            break;
        default:
            break;
    }
    
    return retstate;
}

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_get_caps
         
DESCRIPTION:         This function gets the caps to be set on the source pad.
                     

ARGUMENTS PASSED:    
        v4l_src     - 
         
RETURN VALUE:       Returns the caps to be set.
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstCaps *
mfw_gst_v4lsrc_get_caps (GstBaseSrc * src)
{
  GstCaps *list;
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  GstCaps *capslist;
  GstPadTemplate *src_template = NULL;
  gint i;
  guint32 format;

  switch(v4l_src->pix_fmt)
  {
    case PIX_FMT_RGB565:
      format = GST_MAKE_FOURCC('R', 'G', 'B', 'P');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGB_16);
      break;

    case PIX_FMT_BGR24:
      format = GST_MAKE_FOURCC('B', 'G', 'R', '3');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_BGR);
      break;

    case PIX_FMT_RGB24:
      format = GST_MAKE_FOURCC('R', 'G', 'B', '3');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGB);
      break;

    case PIX_FMT_BGR32:
      format = GST_MAKE_FOURCC('B', 'G', 'R', '4');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_BGRx);
      break;

    case PIX_FMT_RGB32:
      format = GST_MAKE_FOURCC('R', 'G', 'B', '4');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGBx);
      break;

    case PIX_FMT_YUV422P:
      format = GST_MAKE_FOURCC('4', '2', '2', 'P');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("422P"));
      break;

    case PIX_FMT_UYVY:
      format = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("UYVY"));
      break;

    case PIX_FMT_YUYV:
      format = GST_MAKE_FOURCC('Y', 'U', 'Y', 'V');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("YUY2"));
      break;

    case PIX_FMT_YUV420: /* YU12 */
      format = GST_MAKE_FOURCC('Y', 'U', '1', '2');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("I420"));
      break;

    case PIX_FMT_NV12:
      format = GST_MAKE_FOURCC ('N', 'V', '1', '2');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("NV12"));
      break;

    default: /* Unlikely, default color format UYVY is set instead */
      format = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("UYVY"));
      break;
  }

  return capslist;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_fixate
         
DESCRIPTION:         Fixes the Caps on the source pad
                     

ARGUMENTS PASSED:    v4l_src     - 
        
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_fixate (GstPad * pad, GstCaps * caps)
{

  gint i = 0;
  GstStructure *structure = NULL;
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (gst_pad_get_parent (pad));
  guint32 fourcc;

  switch(v4l_src->pix_fmt)
  {
    case PIX_FMT_RGB565:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', 'P'); 
      break;

    case PIX_FMT_BGR24:
      fourcc = GST_MAKE_FOURCC('B', 'G', 'R', '3'); 
      break;

    case PIX_FMT_RGB24:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', '3'); 
      break;

    case PIX_FMT_BGR32:
      fourcc = GST_MAKE_FOURCC('B', 'G', 'R', '4');
      break;

    case PIX_FMT_RGB32:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', '4');
      break;

    case PIX_FMT_YUV422P:
      fourcc = GST_MAKE_FOURCC('4', '2', '2', 'P');
      break;

    case PIX_FMT_UYVY:
      fourcc = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      break;

    case PIX_FMT_YUYV:
      fourcc = GST_MAKE_FOURCC('Y', 'U', 'Y', 'V');
      break;

    case PIX_FMT_YUV420: /* YU12 */
      fourcc = GST_MAKE_FOURCC('Y', 'U', '1', '2'); 
      break;

    case PIX_FMT_NV12:
      fourcc = GST_MAKE_FOURCC ('N', 'V', '1', '2');
      break;

    default: /* Unlikely, default color format UYVY is set instead */
      fourcc = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      break;	
  }   

  const GValue *v = NULL;
  for (i = 0; i < gst_caps_get_size (caps); ++i) {
    structure = gst_caps_get_structure (caps, i);
    gst_structure_fixate_field_nearest_int (structure, "width",
        v4l_src->capture_width);
    gst_structure_fixate_field_nearest_int (structure, "height",
        v4l_src->capture_height);
    gst_structure_fixate_field_nearest_fraction (structure, "framerate",
        v4l_src->fps_out_n, v4l_src->fps_out_d);
    gst_structure_set (structure, "pixel-aspect-ratio", GST_TYPE_FRACTION,
        1, 1, NULL);

    gst_structure_set (structure, "format", GST_TYPE_FOURCC, fourcc, NULL);
  }
  GST_INFO ("capture: %d, %d, fixrate :%s\n", v4l_src->capture_width,
      v4l_src->capture_height, gst_caps_to_string (caps));
  gst_object_unref (v4l_src);

}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_init   
        
DESCRIPTION:     create the pad template that has been registered with the 
                element class in the _base_init and do library table 
                initialization      

ARGUMENTS PASSED:
        context  -    pointer to v4lsrc element structure      
  
RETURN VALUE:       None
      
PRE-CONDITIONS:     _base_init and _class_init are called 
 
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_init (MFWGstV4LSrc * v4l_src, MFWGstV4LSrcClass * klass)
{
  v4l_src->capture_width = 640;
  v4l_src->capture_height = 480;
  v4l_src->fps_n = 30;
  v4l_src->fps_d = 1;
  v4l_src->fps_out_n = 30;
  v4l_src->fps_out_d = 1;
  v4l_src->fd_v4l = -1;
  v4l_src->count = 0;
  v4l_src->buffer_size = 0;
  v4l_src->offset = 0;
  v4l_src->rotate = 0;
  v4l_src->hflip = FALSE;
  v4l_src->vflip = FALSE;
  v4l_src->preview = FALSE;
  v4l_src->preview_width = 160;
  v4l_src->preview_height = 128;
  v4l_src->preview_top = 0;
  v4l_src->preview_left = 0;
  v4l_src->input = 0;
  v4l_src->input_flag = FALSE;
  v4l_src->g_display_lcd = 0;
  v4l_src->queue_size = DEFAULT_QUEUE_SIZE;
  v4l_src->start = FALSE;
  v4l_src->stop = FALSE;
  v4l_src->pix_fmt = PIX_FMT_UYVY; /* Enhanced to support diff RGB and YUV color-formats */
  v4l_src->capture_start_flag = FALSE; /* Enhanced to support diff RGB and YUV color-formats */
  v4l_src->brightness = DEFAULT_BRIGHTNESS_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->contrast = DEFAULT_CONTRAST_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->saturation = DEFAULT_SATURATION_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->hue = DEFAULT_HUE_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->brightness_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->contrast_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->saturation_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->hue_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->v4l2_buf_flag_err = FALSE; /* Enhanced to support error handling signal */
  v4l_src->crop_left = 0; /* Enhanced to support cropping feature */
  v4l_src->crop_top = 0; /* Enhanced to support cropping feature */
  v4l_src->preview_galpha = 255;
  v4l_src->preview_color_key_en = FALSE;
  v4l_src->preview_color_key = 0;
  v4l_src->crop_width = 0; /* Enhanced to support cropping feature */
  v4l_src->crop_height = 0; /* Enhanced to support cropping feature */
  v4l_src->motion = DEFAULT_MOTION_VAL; /* Enhanced to set deinterlace motion compensation parameters */
  v4l_src->motion_set_flag = FALSE; /* Enhanced to set deinterlace motion compensation parameters */
  v4l_src->chroma_blue = DEFAULT_CHROMA_BLUE_VAL; /* Enhanced to support chroma components */
  v4l_src->chroma_red = DEFAULT_CHROMA_RED_VAL; /* Enhanced to support chroma components */
  v4l_src->cb_set_flag = FALSE; /* Enhanced to support chroma components */
  v4l_src->cr_set_flag = FALSE; /* Enhanced to support chroma components */

  v4l_src->devicename = g_strdup ("/dev/video0");
  v4l_src->buf_pools = g_malloc (sizeof (GstBuffer *) * v4l_src->queue_size);
  v4l_src->fist_frame_after_start = FALSE;
  v4l_src->stream_good_buf_only = FALSE;
  gst_pad_set_fixatecaps_function (GST_BASE_SRC_PAD (v4l_src),
      mfw_gst_v4lsrc_fixate);
  gst_base_src_set_live (GST_BASE_SRC (v4l_src), TRUE);

#define MFW_GST_V4LSRC_PLUGIN VERSION
  PRINT_PLUGIN_VERSION (MFW_GST_V4LSRC_PLUGIN);
  return;
}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_class_init    
        
DESCRIPTION:     Initialise the class only once (specifying what signals,
                arguments and virtual functions the class has and setting up 
                global state)    
     

ARGUMENTS PASSED:
       klass   -   pointer to mp3decoder element class
        
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_class_init (MFWGstV4LSrcClass * klass)
{

  GObjectClass *gobject_class;
  GstBaseSrcClass *basesrc_class;
  GstPushSrcClass *pushsrc_class;
  GstElementClass *element_class;

  gobject_class = (GObjectClass *) klass;
  element_class = (GstElementClass *) klass;
  basesrc_class = (GstBaseSrcClass *) klass;
  pushsrc_class = (GstPushSrcClass *) klass;


  gobject_class->set_property = mfw_gst_v4lsrc_set_property;
  gobject_class->get_property = mfw_gst_v4lsrc_get_property;
  element_class->change_state = mfw_gst_v4lsrc_change_state;

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_WIDTH,
      g_param_spec_int ("capture-width",
		  "capture_width",
		  "gets the width of the image to be captured",
		  16, G_MAXINT, 640, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_HEIGHT,
      g_param_spec_int ("capture-height",
		  "capture_height",
		  "gets the height of the image to be captured",
		  16, G_MAXINT, 480, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_INPUT,
      g_param_spec_int ("input",
          "input",
          "set analog video inputs \n"
          "\t\t\t\tinput=0: Composite on Ain1 \n"
          "\t\t\t\tinput=1: Composite on Ain2 \n"
          "\t\t\t\tinput=2: Composite on Ain3 \n"
          "\t\t\t\tinput=3: Composite on Ain4 \n"
          "\t\t\t\tinput=4: Composite on Ain5 \n"
          "\t\t\t\tinput=5: Composite on Ain6",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_WIDTH,
      g_param_spec_int ("preview-width",
          "preview_width",
          "gets the width of the image to be displayed for preview. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          16, 1920, 176, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_HEIGHT,
      g_param_spec_int ("preview-height",
          "preview_height",
          "gets the height of the image to be displayed for preview. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          16, 1080, 144, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_TOP,
      g_param_spec_int ("preview-top",
          "preview_top",
          "gets the top pixel offset at which the preview should start. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_LEFT,
      g_param_spec_int ("preview-left",
          "preview_left",
          "gets the left pixel offset at which the preview should start. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_GALPHA,
      g_param_spec_int ("galpha", "galpha",
                        "sets preview overlay global alpha (0 disables global alpha)",
                        0, 255, 255, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_COLOR_KEY,
      g_param_spec_uint ("color-key", "color-key",
                         "sets preview overlay color key in RGB24 format",
                         0, G_MAXUINT, 0, G_PARAM_READWRITE));

  /* 
   * FixME: The overlay channel will cause v4l error:
   * v4l2 capture: mxc_v4l_dqueue timeout enc_counter 0 error 
   * disable it 
   */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW,
      g_param_spec_boolean ("preview", "Preview",
          "enable the preview of capture, it will directly pass the data to display",
          FALSE, G_PARAM_READWRITE));


  g_object_class_install_property (gobject_class, MFW_V4L_SRC_ROTATE,
      g_param_spec_int ("rotate",
			"Rotate",
			"gets the values by which the camera rotation "
			"angle can be specified in degrees",
			0, 270, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_HFLIP,
      g_param_spec_boolean ("hflip", "hflip",
          "enable horizontal flip", FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_VFLIP,
      g_param_spec_boolean ("vflip", "vflip",
          "enable vertical flip", FALSE, G_PARAM_READWRITE));

  /* Enhanced to support cropping feature */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_LEFT,
      g_param_spec_int ("crop-left",
          "crop-left",
          "number of pixel to crop from the left",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_TOP,
      g_param_spec_int ("crop-top",
          "crop-top",
          "number of pixel to crop from the top",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_WIDTH,
      g_param_spec_int ("crop-width",
          "crop-width",
          "image width after cropping (required to be set to enable cropping)",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_HEIGHT,
      g_param_spec_int ("crop-height",
          "crop-height",
          "image height after cropping (required to be set to enable cropping)",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_FRAMERATE_NUM,
      g_param_spec_int ("fps-n",
          "fps_n",
          "gets the numerator of the framerate at which"
          "the input stream is to be captured",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_FRAMERATE_DEN,
      g_param_spec_int ("fps-d",
          "fps_d",
          "gets the denominator of the framerate at which"
          "the input stream is to be captured",
          1, G_MAXINT, 1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_OUTPUT_FRAMERATE_NUM,
      g_param_spec_int ("fps-out-n",
          "fps_out_n",
          "gets the numerator of the framerate at which"
          "the output stream is available after de-interlacing",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_OUTPUT_FRAMERATE_DEN,
      g_param_spec_int ("fps-out-d",
          "fps_out_d",
          "gets the denominator of the framerate at which"
          "the output stream is available after de-interlacing",
          1, G_MAXINT, 1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_DEVICE,
      g_param_spec_string ("device", "Device", "Device location",
          NULL, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_QUEUE_SIZE,
      g_param_spec_int ("queue-size",
          "queue-size",
          "v4l2 request buffer number", 0, G_MAXINT, 5, G_PARAM_READWRITE));

/* Enhanced to support diff RGB and YUV color-formats */  
  g_object_class_install_property(gobject_class, MFW_V4L_SRC_PIXEL_FMT, 
      g_param_spec_uint("pix-fmt", "Pixel format", 
          "Support different YUV and RGB pixel/color formats \n"
          "\t\t\t\tpix-fmt=0 (PIX_FMT_RGB565)  : Capture RGB565 image \n"
          "\t\t\t\tpix-fmt=1 (PIX_FMT_BGR24)   : Capture BGR24 image \n"
          "\t\t\t\tpix-fmt=2 (PIX_FMT_RGB24)   : Capture RGB24 image \n"
          "\t\t\t\tpix-fmt=3 (PIX_FMT_BGR32)   : Capture BGR32 image \n"
          "\t\t\t\tpix-fmt=4 (PIX_FMT_RGB32)   : Capture RGB32 image \n"
          "\t\t\t\tpix-fmt=5 (PIX_FMT_YUV422P) : Capture YUV422P image \n"
          "\t\t\t\tpix-fmt=6 (PIX_FMT_UYVY)    : Capture UYVY image \n"
          "\t\t\t\tpix-fmt=7 (PIX_FMT_YUYV)    : Capture YUYV image \n"
          "\t\t\t\tpix-fmt=8 (PIX_FMT_YUV420)  : Capture YUV420 image \n"                    
          "\t\t\t\tpix-fmt=9 (PIX_FMT_NV12)    : Capture NV12 image ",          
          0, G_MAX_PIX_FMT, 0, G_PARAM_READWRITE));
          

/* Enhanced to support picture enhancement parameters */
  g_object_class_install_property(gobject_class, MFW_V4L_SRC_BRIGHTNESS,
          g_param_spec_int("brightness", "brightness parameter",
              "Brightness parameter configurable support",
              MIN_BRIGHTNESS_VAL, MAX_BRIGHTNESS_VAL,
              DEFAULT_BRIGHTNESS_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_CONTRAST,
          g_param_spec_int("contrast", "contrast parameter",
              "Contrast parameter configurable support", MIN_CONTRAST_VAL,
              MAX_CONTRAST_VAL, DEFAULT_CONTRAST_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_SATURATION,
          g_param_spec_int("saturation", "saturation parameter",
              "Saturation parameter configurable support", MIN_SATURATION_VAL,
              MAX_SATURATION_VAL, DEFAULT_SATURATION_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_HUE,
          g_param_spec_int("hue", "hue parameter",
              "Hue parameter configurable support", MIN_HUE_VAL,
              MAX_HUE_VAL, DEFAULT_HUE_VAL, G_PARAM_READWRITE));

/* Enhanced to set deinterlace motion compensation parameters */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_MOTION,
          g_param_spec_int ("motion","motion parameter",
              "The deinterlace motion parameter setting \n"
              "\t\t\t\tmotion=0:  no motion \n"
              "\t\t\t\tmotion=1:  low motion \n"
              "\t\t\t\tmotion=2:  medium motion \n"
              "\t\t\t\tmotion=3:  high motion",
              0, G_MAX_MOTION_VAL, 0, G_PARAM_READWRITE));

/* Enhanced to support chroma components */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CHROMA_BLUE,
		  g_param_spec_int ("chroma-blue","chroma blue parameter",
			  "Chroma Blue parameter setting",
			  MIN_CHROMA_BLUE_VAL, MAX_CHROMA_BLUE_VAL,
			  DEFAULT_CHROMA_BLUE_VAL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CHROMA_RED,
		  g_param_spec_int ("chroma-red","chroma red parameter",
			  "Chroma Red parameter setting",
			  MIN_CHROMA_RED_VAL, MAX_CHROMA_RED_VAL,
			  DEFAULT_CHROMA_RED_VAL, G_PARAM_READWRITE));

/* Avoid streaming of bad buffers */
  g_object_class_install_property (gobject_class, MFW_V4L_STREAM_GOOD_BUFFER,
		  g_param_spec_boolean ("stream-good-buf-only","stream good buffers only",
			  "Setting to avoid streaming of bad buffers\n"
                          "\t\t\t\tstream-good-buf-only=0:  all buffers are pushed to next element \n"
                          "\t\t\t\tstream-good-buf-only=1:  Only good buffers are pushed to next element \n",
			  FALSE, G_PARAM_READWRITE));

  /**
   * Add signal. Check for V4L2_BUF_FLAG_ERROR flag and emit
   * signal if flag is set
   */
  gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_ERROR] =
      g_signal_new ("v4l-buf-flag-error",
          G_TYPE_FROM_CLASS (klass),
          G_SIGNAL_RUN_FIRST,
          G_STRUCT_OFFSET (MFWGstV4LSrcClass, v4l_buf_flag_error),
          NULL, NULL,
          gst_marshal_VOID__OBJECT,
          G_TYPE_NONE, 1,
          GST_TYPE_PAD);

  /**
   * Add SIGNAL_V4L2_BUF_FLAG_OK signal to notify the Gstreamer framework
   */
  gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_OK] =
      g_signal_new ("v4l-buf-flag-ok",
          G_TYPE_FROM_CLASS (klass),
          G_SIGNAL_RUN_FIRST,
          G_STRUCT_OFFSET (MFWGstV4LSrcClass, v4l_buf_flag_ok),
          NULL, NULL,
          gst_marshal_VOID__OBJECT,
          G_TYPE_NONE, 1,
          GST_TYPE_PAD);

  basesrc_class->get_caps = mfw_gst_v4lsrc_get_caps;
  basesrc_class->set_caps = mfw_gst_v4lsrc_set_caps;
  basesrc_class->start = mfw_gst_v4lsrc_start;
  basesrc_class->stop = mfw_gst_v4lsrc_stop;
  basesrc_class->unlock = mfw_gst_v4lsrc_unlock;
  pushsrc_class->create = mfw_gst_v4lsrc_create;
  return;
}


/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_base_init   
        
DESCRIPTION:     v4l source element details are registered with the plugin during
                _base_init ,This function will initialise the class and child 
                class properties during each new child class creation       

ARGUMENTS PASSED:
        Klass   -   void pointer
  
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  FSL_GST_ELEMENT_SET_DETAIL_SIMPLE (element_class, "v4l2 based camera src",
      "Src/Video", "Capture videos by using csi camera");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_factory));

  GST_DEBUG_CATEGORY_INIT (mfw_gst_v4lsrc_debug, "mfw_v4lsrc", 0,
      "V4L2 video src element");

  return;
}

/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or 
                element is loaded and information returned by this function 
                will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded 
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any 
        dependency correctly

PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static gboolean
plugin_init (GstPlugin * plugin)
{
  if (!gst_element_register (plugin, "mfw_v4lsrc", GST_RANK_PRIMARY,
          MFW_GST_TYPE_V4LSRC))
    return FALSE;

  return TRUE;
}

/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/

FSL_GST_PLUGIN_DEFINE ("v4lsrc", "v4l2-based csi camera video src",
    plugin_init);
